import Foundation

/// App Store Compliance utility for managing medical disclaimer and regulatory requirements
struct AppStoreCompliance {
    
    // MARK: - Disclaimer Text Constants
    
    static let shortDisclaimer = """
    This app is for entertainment and educational purposes only. 
    It is not a medical device or diagnostic tool. 
    Always consult healthcare professionals for medical advice.
    """
    
    static let medicalDisclaimerText = """
    IMPORTANT MEDICAL DISCLAIMER
    
    This application is designed for entertainment and educational purposes only. 
    It is NOT a medical device, diagnostic tool, or health assessment instrument.
    
    The calculations, predictions, and information provided by this app do not 
    constitute medical advice, diagnosis, or treatment recommendations.
    
    Always consult with qualified healthcare professionals for medical advice, 
    diagnosis, and treatment decisions.
    
    All predictions are based on statistical models and general population data. 
    Individual results may vary significantly and should not be used for any 
    medical or health-related decisions.
    """
    
    static let appStoreGuidelines = """
    This app complies with Apple's App Store Review Guidelines:
    
    • Does not claim to diagnose, cure, treat, or prevent any medical condition
    • Clearly states entertainment and educational purpose only
    • Includes prominent medical disclaimer
    • Does not encourage dangerous behavior
    • Protects user privacy and data security
    • Provides appropriate content warnings
    """
    
    // MARK: - Compliance Checks
    
    /// Checks if the user has accepted the current version of the disclaimer
    static func hasValidDisclaimerAcceptance() -> Bool {
        let hasAccepted = UserDefaults.standard.bool(forKey: "hasAcceptedMedicalDisclaimer")
        
        guard hasAccepted,
              let acceptanceDate = UserDefaults.standard.object(forKey: "disclaimerAcceptanceDate") as? Date else {
            return false
        }
        
        // Check if disclaimer acceptance is still valid (within 90 days)
        let daysSinceAcceptance = Calendar.current.dateComponents([.day], from: acceptanceDate, to: Date()).day ?? 0
        return daysSinceAcceptance <= 90
    }
    
    /// Records disclaimer acceptance with timestamp
    static func recordDisclaimerAcceptance() {
        UserDefaults.standard.set(true, forKey: "hasAcceptedMedicalDisclaimer")
        UserDefaults.standard.set(Date(), forKey: "disclaimerAcceptanceDate")
        UserDefaults.standard.set(currentDisclaimerVersion, forKey: "acceptedDisclaimerVersion")
        
        // Create audit trail
        let acceptanceRecord: [String: Any] = [
            "timestamp": Date().timeIntervalSince1970,
            "version": currentDisclaimerVersion,
            "event": "medical_disclaimer_accepted"
        ]
        
        UserDefaults.standard.set(acceptanceRecord, forKey: "disclaimerAcceptanceRecord")
    }
    
    /// Current disclaimer version for tracking changes
    static let currentDisclaimerVersion = "1.0.0"
    
    // MARK: - Privacy Compliance
    
    /// Ensures data processing complies with privacy regulations
    static func isPrivacyCompliant() -> Bool {
        // Check that sensitive data is processed locally
        // Verify no unauthorized data transmission
        // Confirm user has control over their data
        return true // Implementation would include actual privacy checks
    }
    
    /// Provides user with data control options
    static func getUserDataControlOptions() -> [String] {
        return [
            "View stored data",
            "Export personal data",
            "Delete all data",
            "Manage privacy settings"
        ]
    }
    
    // MARK: - Content Rating Compliance
    
    /// Ensures content is appropriate for intended audience
    static let contentRating = "12+" // Due to mortality-related content
    
    static let contentWarnings = [
        "Contains mortality-related calculations",
        "May cause anxiety in some users",
        "Not suitable for users with health anxiety",
        "Educational/entertainment purpose only"
    ]
    
    // MARK: - Emergency Information
    
    static let emergencyDisclaimer = """
    This app is not designed for emergency situations. 
    If you are experiencing a medical emergency, contact emergency services immediately:
    
    • US: 911
    • UK: 999
    • EU: 112
    • Or your local emergency number
    """
    
    // MARK: - Data Retention Policy
    
    static let dataRetentionPolicy = """
    Data Retention Policy:
    
    • All health data is stored locally on your device
    • No data is transmitted to external servers
    • You can delete all data at any time
    • Data is automatically encrypted
    • App removal deletes all associated data
    """
    
    // MARK: - Legal Disclaimer
    
    static let legalDisclaimer = """
    LEGAL DISCLAIMER
    
    The developers, publishers, and distributors of this application disclaim 
    all liability for any decisions made based on the app's output, calculations, 
    or recommendations.
    
    Use of this application is entirely at your own risk. The app is provided 
    "as is" without warranty of any kind, either expressed or implied.
    
    This disclaimer may be updated periodically. Continued use of the app after 
    updates constitutes acceptance of the revised disclaimer.
    """
    
    // MARK: - Validation Methods
    
    /// Validates that all compliance requirements are met
    static func validateCompliance() -> ComplianceStatus {
        var issues: [String] = []
        
        if !hasValidDisclaimerAcceptance() {
            issues.append("Medical disclaimer not accepted or expired")
        }
        
        if !isPrivacyCompliant() {
            issues.append("Privacy compliance issues detected")
        }
        
        return ComplianceStatus(
            isCompliant: issues.isEmpty,
            issues: issues,
            lastChecked: Date()
        )
    }
}

// MARK: - Compliance Status Model

struct ComplianceStatus {
    let isCompliant: Bool
    let issues: [String]
    let lastChecked: Date
    
    var summary: String {
        if isCompliant {
            return "All compliance requirements met"
        } else {
            return "Compliance issues found: \(issues.joined(separator: ", "))"
        }
    }
}

// MARK: - Disclaimer Manager Extension

extension DisclaimerManager {
    
    /// Checks if disclaimer needs to be shown based on App Store compliance
    func shouldShowDisclaimer() -> Bool {
        return !AppStoreCompliance.hasValidDisclaimerAcceptance()
    }
    
    /// Records disclaimer acceptance with full compliance tracking
    func acceptDisclaimerWithCompliance() {
        AppStoreCompliance.recordDisclaimerAcceptance()
        hasAcceptedDisclaimer = true
    }
    
    /// Gets current compliance status
    func getComplianceStatus() -> ComplianceStatus {
        return AppStoreCompliance.validateCompliance()
    }
}